import { useInternetIdentity } from '../hooks/useInternetIdentity';
import { useSubmitServiceRequest } from '../hooks/useQueries';
import { ServiceType, RequestStatus, ExternalBlob } from '../backend';
import ServiceForm from '../components/ServiceForm';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { useNavigate } from '@tanstack/react-router';
import { AlertCircle } from 'lucide-react';

export default function SocietyTrustPage() {
  const { identity } = useInternetIdentity();
  const navigate = useNavigate();
  const submitRequest = useSubmitServiceRequest();

  const handleSubmit = async (details: string, documents: ExternalBlob[]) => {
    if (!identity) return;

    const request = {
      id: `society-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
      user: identity.getPrincipal(),
      serviceType: ServiceType.societyTrustRegistration,
      details,
      documents,
      status: RequestStatus.submitted,
      fee: BigInt(5000),
      createdAt: BigInt(Date.now() * 1000000),
    };

    await submitRequest.mutateAsync(request);
  };

  if (!identity) {
    return (
      <div className="container mx-auto px-4 py-12">
        <Card className="max-w-2xl mx-auto">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <AlertCircle className="h-5 w-5 text-primary" />
              Login Required
            </CardTitle>
            <CardDescription>
              Please login to access Society/Trust registration services
            </CardDescription>
          </CardHeader>
          <CardContent>
            <Button onClick={() => navigate({ to: '/' })}>
              Go to Home
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="container mx-auto px-4 py-8">
      <div className="max-w-3xl mx-auto">
        <div className="mb-8">
          <h1 className="text-3xl font-bold mb-2">🏛️ Society / Trust Registration</h1>
          <p className="text-muted-foreground">
            Register your society or trust with renewal and compliance services
          </p>
        </div>

        <ServiceForm
          title="Society/Trust Registration"
          description="Register a new society/trust or renew existing registration"
          serviceType={ServiceType.societyTrustRegistration}
          fee={5000}
          onSubmit={handleSubmit}
          fields={[
            { name: 'serviceType', label: 'Service Type', required: true, placeholder: 'New Registration / Renewal / Return Filing' },
            { name: 'organizationType', label: 'Organization Type', required: true, placeholder: 'Society / Trust / Section 8 Company' },
            { name: 'organizationName', label: 'Organization Name', required: true, placeholder: 'Proposed/Existing name' },
            { name: 'objectives', label: 'Objectives', type: 'textarea', required: true, placeholder: 'Main objectives of the organization' },
            { name: 'members', label: 'Number of Members', required: true, placeholder: 'Total members/trustees' },
            { name: 'registeredAddress', label: 'Registered Address', type: 'textarea', required: true, placeholder: 'Complete registered address' },
          ]}
        />
      </div>
    </div>
  );
}
