import { useInternetIdentity } from '../hooks/useInternetIdentity';
import { useSubmitServiceRequest } from '../hooks/useQueries';
import { ServiceType, RequestStatus, ExternalBlob } from '../backend';
import ServiceForm from '../components/ServiceForm';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { useNavigate } from '@tanstack/react-router';
import { AlertCircle } from 'lucide-react';

export default function RTIServicesPage() {
  const { identity } = useInternetIdentity();
  const navigate = useNavigate();
  const submitRequest = useSubmitServiceRequest();

  const handleSubmit = async (details: string, documents: ExternalBlob[]) => {
    if (!identity) return;

    const request = {
      id: `rti-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
      user: identity.getPrincipal(),
      serviceType: ServiceType.rtiServices,
      details,
      documents,
      status: RequestStatus.submitted,
      fee: BigInt(500),
      createdAt: BigInt(Date.now() * 1000000),
    };

    await submitRequest.mutateAsync(request);
  };

  if (!identity) {
    return (
      <div className="container mx-auto px-4 py-12">
        <Card className="max-w-2xl mx-auto">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <AlertCircle className="h-5 w-5 text-primary" />
              Login Required
            </CardTitle>
            <CardDescription>
              Please login to access RTI services
            </CardDescription>
          </CardHeader>
          <CardContent>
            <Button onClick={() => navigate({ to: '/' })}>
              Go to Home
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="container mx-auto px-4 py-8">
      <div className="max-w-3xl mx-auto">
        <div className="mb-8">
          <h1 className="text-3xl font-bold mb-2">🧾 RTI Services</h1>
          <p className="text-muted-foreground">
            File Right to Information requests with government departments
          </p>
        </div>

        <ServiceForm
          title="RTI Application"
          description="Submit your Right to Information request"
          serviceType={ServiceType.rtiServices}
          fee={500}
          onSubmit={handleSubmit}
          fields={[
            { name: 'department', label: 'Government Department', required: true, placeholder: 'Name of the department' },
            { name: 'informationSought', label: 'Information Sought', type: 'textarea', required: true, placeholder: 'Describe the information you are seeking' },
            { name: 'purpose', label: 'Purpose of Information', required: true, placeholder: 'Why do you need this information?' },
            { name: 'address', label: 'Postal Address', type: 'textarea', required: true, placeholder: 'Your complete postal address' },
          ]}
        />
      </div>
    </div>
  );
}
