import { useInternetIdentity } from '../hooks/useInternetIdentity';
import { useGetCallerUserProfile, useGetUserServiceRequests, useGetUserPaymentRecords } from '../hooks/useQueries';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { useNavigate } from '@tanstack/react-router';
import { User, FileText, CreditCard, AlertCircle, Loader2 } from 'lucide-react';
import { RequestStatus, PaymentStatus } from '../backend';
import UpiQrCode from '../components/UpiQrCode';

export default function ProfilePage() {
  const { identity } = useInternetIdentity();
  const navigate = useNavigate();
  const { data: userProfile, isLoading: profileLoading } = useGetCallerUserProfile();
  const { data: serviceRequests = [], isLoading: requestsLoading } = useGetUserServiceRequests();
  const { data: paymentRecords = [], isLoading: paymentsLoading } = useGetUserPaymentRecords();

  if (!identity) {
    return (
      <div className="container mx-auto px-4 py-12">
        <Card className="max-w-2xl mx-auto">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <AlertCircle className="h-5 w-5 text-primary" />
              Login Required
            </CardTitle>
            <CardDescription>
              Please login to view your profile
            </CardDescription>
          </CardHeader>
          <CardContent>
            <Button onClick={() => navigate({ to: '/' })}>
              Go to Home
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  if (profileLoading) {
    return (
      <div className="container mx-auto px-4 py-12 flex justify-center">
        <Loader2 className="h-8 w-8 animate-spin text-primary" />
      </div>
    );
  }

  const getStatusBadge = (status: RequestStatus) => {
    const variants: Record<RequestStatus, 'default' | 'secondary' | 'outline' | 'destructive'> = {
      [RequestStatus.submitted]: 'secondary',
      [RequestStatus.underReview]: 'default',
      [RequestStatus.approved]: 'outline',
      [RequestStatus.completed]: 'default',
    };
    return <Badge variant={variants[status]}>{status}</Badge>;
  };

  const getPaymentStatusBadge = (status: PaymentStatus) => {
    const variants: Record<PaymentStatus, 'default' | 'secondary' | 'outline' | 'destructive'> = {
      [PaymentStatus.pending]: 'secondary',
      [PaymentStatus.verified]: 'default',
      [PaymentStatus.rejected]: 'destructive',
    };
    return <Badge variant={variants[status]}>{status}</Badge>;
  };

  const formatDate = (timestamp: bigint) => {
    const date = new Date(Number(timestamp) / 1000000);
    return date.toLocaleDateString('en-IN', { year: 'numeric', month: 'short', day: 'numeric' });
  };

  const getServiceTypeName = (type: string) => {
    const names: Record<string, string> = {
      legalHelp: 'Legal Help',
      rtiServices: 'RTI Services',
      panIncomeTax: 'PAN & Income Tax',
      gstServices: 'GST Services',
      pfEsiServices: 'PF & ESI Services',
      societyTrustRegistration: 'Society/Trust Registration',
      accountingBookkeeping: 'Accounting & Bookkeeping',
    };
    return names[type] || type;
  };

  return (
    <div className="container mx-auto px-4 py-8">
      <div className="max-w-6xl mx-auto">
        <div className="mb-8">
          <h1 className="text-3xl font-bold mb-2">My Profile</h1>
          <p className="text-muted-foreground">
            Manage your profile and track your applications
          </p>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2 text-lg">
                <User className="h-5 w-5" />
                Profile Info
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-2">
              <div>
                <p className="text-sm text-muted-foreground">Name</p>
                <p className="font-medium">{userProfile?.name}</p>
              </div>
              <div>
                <p className="text-sm text-muted-foreground">Contact</p>
                <p className="font-medium">{userProfile?.contact}</p>
              </div>
              <div>
                <p className="text-sm text-muted-foreground">Email</p>
                <p className="font-medium">{userProfile?.email}</p>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2 text-lg">
                <FileText className="h-5 w-5" />
                Applications
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-primary">{serviceRequests.length}</div>
              <p className="text-sm text-muted-foreground">Total applications submitted</p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2 text-lg">
                <CreditCard className="h-5 w-5" />
                Payments
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-primary">{paymentRecords.length}</div>
              <p className="text-sm text-muted-foreground">Payment records</p>
            </CardContent>
          </Card>
        </div>

        <Tabs defaultValue="applications" className="w-full">
          <TabsList className="grid w-full grid-cols-3">
            <TabsTrigger value="applications">My Applications</TabsTrigger>
            <TabsTrigger value="payments">Payment Records</TabsTrigger>
            <TabsTrigger value="payment-qr">Payment QR</TabsTrigger>
          </TabsList>

          <TabsContent value="applications" className="mt-6">
            <Card>
              <CardHeader>
                <CardTitle>Service Applications</CardTitle>
                <CardDescription>Track the status of your submitted applications</CardDescription>
              </CardHeader>
              <CardContent>
                {requestsLoading ? (
                  <div className="flex justify-center py-8">
                    <Loader2 className="h-6 w-6 animate-spin text-primary" />
                  </div>
                ) : serviceRequests.length === 0 ? (
                  <div className="text-center py-8 text-muted-foreground">
                    No applications submitted yet
                  </div>
                ) : (
                  <div className="overflow-x-auto">
                    <Table>
                      <TableHeader>
                        <TableRow>
                          <TableHead>Service Type</TableHead>
                          <TableHead>Date</TableHead>
                          <TableHead>Fee</TableHead>
                          <TableHead>Status</TableHead>
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {serviceRequests.map((request) => (
                          <TableRow key={request.id}>
                            <TableCell className="font-medium">
                              {getServiceTypeName(request.serviceType)}
                            </TableCell>
                            <TableCell>{formatDate(request.createdAt)}</TableCell>
                            <TableCell>₹{Number(request.fee)}</TableCell>
                            <TableCell>{getStatusBadge(request.status)}</TableCell>
                          </TableRow>
                        ))}
                      </TableBody>
                    </Table>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="payments" className="mt-6">
            <Card>
              <CardHeader>
                <CardTitle>Payment Records</CardTitle>
                <CardDescription>View your payment history and verification status</CardDescription>
              </CardHeader>
              <CardContent>
                {paymentsLoading ? (
                  <div className="flex justify-center py-8">
                    <Loader2 className="h-6 w-6 animate-spin text-primary" />
                  </div>
                ) : paymentRecords.length === 0 ? (
                  <div className="text-center py-8 text-muted-foreground">
                    No payment records found
                  </div>
                ) : (
                  <div className="overflow-x-auto">
                    <Table>
                      <TableHeader>
                        <TableRow>
                          <TableHead>Payment ID</TableHead>
                          <TableHead>Date</TableHead>
                          <TableHead>Amount</TableHead>
                          <TableHead>Status</TableHead>
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {paymentRecords.map((record) => (
                          <TableRow key={record.id}>
                            <TableCell className="font-mono text-sm">{record.id.slice(0, 16)}...</TableCell>
                            <TableCell>{formatDate(record.createdAt)}</TableCell>
                            <TableCell>₹{Number(record.amount) / 100}</TableCell>
                            <TableCell>{getPaymentStatusBadge(record.status)}</TableCell>
                          </TableRow>
                        ))}
                      </TableBody>
                    </Table>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="payment-qr" className="mt-6">
            <div className="max-w-2xl mx-auto">
              <UpiQrCode 
                title="UPI Payment QR Code"
                description="Scan this QR code to make payments for any service"
                showInstructions={true}
              />
              <div className="mt-6 text-center">
                <Button onClick={() => navigate({ to: '/payment' })} size="lg">
                  Go to Payment Page
                </Button>
              </div>
            </div>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  );
}
