import { useState } from 'react';
import { useInternetIdentity } from '../hooks/useInternetIdentity';
import { useUploadPaymentProof } from '../hooks/useQueries';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { useNavigate } from '@tanstack/react-router';
import { AlertCircle, CheckCircle, Loader2, Upload } from 'lucide-react';
import { toast } from 'sonner';
import { PaymentStatus, ExternalBlob } from '../backend';
import UpiQrCode from '../components/UpiQrCode';

export default function PaymentPage() {
  const { identity } = useInternetIdentity();
  const navigate = useNavigate();
  const uploadPayment = useUploadPaymentProof();
  
  const [amount, setAmount] = useState('');
  const [paymentProofFile, setPaymentProofFile] = useState<File | null>(null);
  const [uploading, setUploading] = useState(false);

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      const file = e.target.files[0];
      const ext = file.name.split('.').pop()?.toLowerCase();
      if (['jpg', 'jpeg', 'png', 'pdf'].includes(ext || '')) {
        setPaymentProofFile(file);
      } else {
        toast.error('Please upload a valid image or PDF file');
      }
    }
  };

  const handleMarkAsPaid = async () => {
    if (!identity) {
      toast.error('Please login to mark payment');
      return;
    }

    if (!amount || parseFloat(amount) <= 0) {
      toast.error('Please enter a valid amount');
      return;
    }

    if (!paymentProofFile) {
      toast.error('Please upload payment proof');
      return;
    }

    setUploading(true);

    try {
      const arrayBuffer = await paymentProofFile.arrayBuffer();
      const uint8Array = new Uint8Array(arrayBuffer);
      const proofBlob = ExternalBlob.fromBytes(uint8Array);

      const paymentRecord = {
        id: `payment-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
        user: identity.getPrincipal(),
        amount: BigInt(Math.round(parseFloat(amount) * 100)),
        paymentProof: proofBlob,
        status: PaymentStatus.pending,
        createdAt: BigInt(Date.now() * 1000000),
      };

      await uploadPayment.mutateAsync(paymentRecord);
      
      toast.success('Payment marked as paid! Awaiting verification.');
      setAmount('');
      setPaymentProofFile(null);
      
      setTimeout(() => {
        navigate({ to: '/profile' });
      }, 2000);
    } catch (error) {
      console.error('Payment submission error:', error);
      toast.error('Failed to submit payment proof');
    } finally {
      setUploading(false);
    }
  };

  if (!identity) {
    return (
      <div className="container mx-auto px-4 py-12">
        <Card className="max-w-2xl mx-auto">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <AlertCircle className="h-5 w-5 text-primary" />
              Login Required
            </CardTitle>
            <CardDescription>
              Please login to access payment services
            </CardDescription>
          </CardHeader>
          <CardContent>
            <p className="text-muted-foreground mb-4">
              You need to be logged in to make payments and submit payment proofs.
            </p>
            <Button onClick={() => navigate({ to: '/' })}>
              Go to Home
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="container mx-auto px-4 py-8">
      <div className="max-w-5xl mx-auto">
        <div className="mb-8">
          <h1 className="text-3xl font-bold mb-2">💳 Payment</h1>
          <p className="text-muted-foreground">
            Scan the QR code to make secure UPI payments
          </p>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          <UpiQrCode 
            title="Scan to Pay"
            description="Use any UPI app to scan and pay"
            showInstructions={true}
          />

          <Card>
            <CardHeader>
              <CardTitle>Mark Payment as Paid</CardTitle>
              <CardDescription>
                After making the payment, upload proof and mark as paid
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="space-y-2">
                <Label htmlFor="amount">
                  Amount Paid (₹) <span className="text-destructive">*</span>
                </Label>
                <Input
                  id="amount"
                  type="number"
                  step="0.01"
                  min="0"
                  value={amount}
                  onChange={(e) => setAmount(e.target.value)}
                  placeholder="Enter amount paid"
                  required
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="paymentProof">
                  Payment Proof (Screenshot/Receipt) <span className="text-destructive">*</span>
                </Label>
                <div className="flex items-center gap-2">
                  <Input
                    id="paymentProof"
                    type="file"
                    onChange={handleFileChange}
                    accept=".jpg,.jpeg,.png,.pdf"
                    className="cursor-pointer"
                  />
                  <Upload className="h-5 w-5 text-muted-foreground" />
                </div>
                {paymentProofFile && (
                  <p className="text-sm text-muted-foreground">
                    Selected: {paymentProofFile.name}
                  </p>
                )}
              </div>

              <div className="bg-muted/50 p-4 rounded-lg space-y-2">
                <h4 className="font-medium text-sm flex items-center gap-2">
                  <CheckCircle className="h-4 w-4 text-primary" />
                  Instructions
                </h4>
                <ul className="text-sm text-muted-foreground space-y-1 list-disc list-inside">
                  <li>Scan the QR code using any UPI app</li>
                  <li>Complete the payment</li>
                  <li>Take a screenshot of the payment confirmation</li>
                  <li>Upload the screenshot and enter the amount</li>
                  <li>Click "Mark as Paid" to submit for verification</li>
                </ul>
              </div>

              <Button 
                onClick={handleMarkAsPaid} 
                className="w-full" 
                disabled={uploading || !amount || !paymentProofFile}
              >
                {uploading ? (
                  <>
                    <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                    Submitting...
                  </>
                ) : (
                  'Mark as Paid'
                )}
              </Button>

              <p className="text-xs text-center text-muted-foreground">
                Your payment will be verified by our team within 24 hours
              </p>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}
