import { useInternetIdentity } from '../hooks/useInternetIdentity';
import { useSubmitServiceRequest } from '../hooks/useQueries';
import { ServiceType, RequestStatus, ExternalBlob } from '../backend';
import ServiceForm from '../components/ServiceForm';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { useNavigate } from '@tanstack/react-router';
import { AlertCircle } from 'lucide-react';

export default function PFESIServicesPage() {
  const { identity } = useInternetIdentity();
  const navigate = useNavigate();
  const submitRequest = useSubmitServiceRequest();

  const handleSubmit = async (details: string, documents: ExternalBlob[]) => {
    if (!identity) return;

    const request = {
      id: `pfesi-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
      user: identity.getPrincipal(),
      serviceType: ServiceType.pfEsiServices,
      details,
      documents,
      status: RequestStatus.submitted,
      fee: BigInt(2500),
      createdAt: BigInt(Date.now() * 1000000),
    };

    await submitRequest.mutateAsync(request);
  };

  if (!identity) {
    return (
      <div className="container mx-auto px-4 py-12">
        <Card className="max-w-2xl mx-auto">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <AlertCircle className="h-5 w-5 text-primary" />
              Login Required
            </CardTitle>
            <CardDescription>
              Please login to access PF & ESI services
            </CardDescription>
          </CardHeader>
          <CardContent>
            <Button onClick={() => navigate({ to: '/' })}>
              Go to Home
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="container mx-auto px-4 py-8">
      <div className="max-w-3xl mx-auto">
        <div className="mb-8">
          <h1 className="text-3xl font-bold mb-2">🏢 PF & ESI Services</h1>
          <p className="text-muted-foreground">
            Provident Fund and Employee State Insurance registration and compliance
          </p>
        </div>

        <ServiceForm
          title="PF & ESI Application"
          description="Register for PF/ESI or submit compliance returns"
          serviceType={ServiceType.pfEsiServices}
          fee={2500}
          onSubmit={handleSubmit}
          fields={[
            { name: 'serviceType', label: 'Service Type', required: true, placeholder: 'PF Registration / ESI Registration / Compliance Filing' },
            { name: 'companyName', label: 'Company Name', required: true, placeholder: 'Legal name of the company' },
            { name: 'numberOfEmployees', label: 'Number of Employees', required: true, placeholder: 'Total employee count' },
            { name: 'establishmentType', label: 'Establishment Type', required: true, placeholder: 'Factory / Office / Shop' },
            { name: 'registrationNumber', label: 'Company Registration Number', required: true, placeholder: 'CIN / Registration number' },
            { name: 'address', label: 'Establishment Address', type: 'textarea', required: true, placeholder: 'Complete address' },
          ]}
        />
      </div>
    </div>
  );
}
