import { useInternetIdentity } from '../hooks/useInternetIdentity';
import { useSubmitServiceRequest } from '../hooks/useQueries';
import { ServiceType, RequestStatus, ExternalBlob } from '../backend';
import ServiceForm from '../components/ServiceForm';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { useNavigate } from '@tanstack/react-router';
import { AlertCircle } from 'lucide-react';

export default function PANTaxPage() {
  const { identity } = useInternetIdentity();
  const navigate = useNavigate();
  const submitRequest = useSubmitServiceRequest();

  const handleSubmit = async (details: string, documents: ExternalBlob[]) => {
    if (!identity) return;

    const request = {
      id: `pan-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
      user: identity.getPrincipal(),
      serviceType: ServiceType.panIncomeTax,
      details,
      documents,
      status: RequestStatus.submitted,
      fee: BigInt(1500),
      createdAt: BigInt(Date.now() * 1000000),
    };

    await submitRequest.mutateAsync(request);
  };

  if (!identity) {
    return (
      <div className="container mx-auto px-4 py-12">
        <Card className="max-w-2xl mx-auto">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <AlertCircle className="h-5 w-5 text-primary" />
              Login Required
            </CardTitle>
            <CardDescription>
              Please login to access PAN & Tax services
            </CardDescription>
          </CardHeader>
          <CardContent>
            <Button onClick={() => navigate({ to: '/' })}>
              Go to Home
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="container mx-auto px-4 py-8">
      <div className="max-w-3xl mx-auto">
        <div className="mb-8">
          <h1 className="text-3xl font-bold mb-2">🧍‍♂️ PAN & Income Tax Services</h1>
          <p className="text-muted-foreground">
            PAN card application and Income Tax return filing assistance
          </p>
        </div>

        <ServiceForm
          title="PAN & Tax Application"
          description="Apply for PAN card or file your income tax returns"
          serviceType={ServiceType.panIncomeTax}
          fee={1500}
          onSubmit={handleSubmit}
          fields={[
            { name: 'serviceType', label: 'Service Type', required: true, placeholder: 'PAN Application / Tax Return Filing' },
            { name: 'fullName', label: 'Full Name (as per Aadhaar)', required: true, placeholder: 'Enter your full name' },
            { name: 'dateOfBirth', label: 'Date of Birth', type: 'date', required: true },
            { name: 'aadhaarNumber', label: 'Aadhaar Number', required: true, placeholder: 'Enter 12-digit Aadhaar number' },
            { name: 'fatherName', label: "Father's Name", required: true, placeholder: "Enter father's name" },
            { name: 'address', label: 'Residential Address', type: 'textarea', required: true, placeholder: 'Complete address with pincode' },
          ]}
        />
      </div>
    </div>
  );
}
