import { useInternetIdentity } from '../hooks/useInternetIdentity';
import { useSubmitServiceRequest } from '../hooks/useQueries';
import { ServiceType, RequestStatus, ExternalBlob } from '../backend';
import ServiceForm from '../components/ServiceForm';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { useNavigate } from '@tanstack/react-router';
import { AlertCircle } from 'lucide-react';

export default function LegalHelpPage() {
  const { identity } = useInternetIdentity();
  const navigate = useNavigate();
  const submitRequest = useSubmitServiceRequest();

  const handleSubmit = async (details: string, documents: ExternalBlob[]) => {
    if (!identity) return;

    const request = {
      id: `legal-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
      user: identity.getPrincipal(),
      serviceType: ServiceType.legalHelp,
      details,
      documents,
      status: RequestStatus.submitted,
      fee: BigInt(2000),
      createdAt: BigInt(Date.now() * 1000000),
    };

    await submitRequest.mutateAsync(request);
  };

  if (!identity) {
    return (
      <div className="container mx-auto px-4 py-12">
        <Card className="max-w-2xl mx-auto">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <AlertCircle className="h-5 w-5 text-primary" />
              Login Required
            </CardTitle>
            <CardDescription>
              Please login to access legal help services
            </CardDescription>
          </CardHeader>
          <CardContent>
            <p className="text-muted-foreground mb-4">
              You need to be logged in to submit legal help requests and track your applications.
            </p>
            <Button onClick={() => navigate({ to: '/' })}>
              Go to Home
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="container mx-auto px-4 py-8">
      <div className="max-w-3xl mx-auto">
        <div className="mb-8">
          <h1 className="text-3xl font-bold mb-2">⚖️ Legal Help Services</h1>
          <p className="text-muted-foreground">
            Get expert legal consultation and guidance for your cases
          </p>
        </div>

        <ServiceForm
          title="Legal Consultation Request"
          description="Provide details about your legal matter and our experts will guide you"
          serviceType={ServiceType.legalHelp}
          fee={2000}
          onSubmit={handleSubmit}
          fields={[
            { name: 'caseType', label: 'Type of Legal Matter', required: true, placeholder: 'e.g., Civil, Criminal, Family, Property' },
            { name: 'description', label: 'Case Description', type: 'textarea', required: true, placeholder: 'Describe your legal issue in detail' },
            { name: 'urgency', label: 'Urgency Level', required: true, placeholder: 'e.g., Urgent, Normal, Low Priority' },
            { name: 'preferredContact', label: 'Preferred Contact Method', required: true, placeholder: 'Phone, Email, or WhatsApp' },
          ]}
        />
      </div>
    </div>
  );
}
