import { useInternetIdentity } from '../hooks/useInternetIdentity';
import { useSubmitServiceRequest } from '../hooks/useQueries';
import { ServiceType, RequestStatus, ExternalBlob } from '../backend';
import ServiceForm from '../components/ServiceForm';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { useNavigate } from '@tanstack/react-router';
import { AlertCircle } from 'lucide-react';

export default function GSTServicesPage() {
  const { identity } = useInternetIdentity();
  const navigate = useNavigate();
  const submitRequest = useSubmitServiceRequest();

  const handleSubmit = async (details: string, documents: ExternalBlob[]) => {
    if (!identity) return;

    const request = {
      id: `gst-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
      user: identity.getPrincipal(),
      serviceType: ServiceType.gstServices,
      details,
      documents,
      status: RequestStatus.submitted,
      fee: BigInt(3000),
      createdAt: BigInt(Date.now() * 1000000),
    };

    await submitRequest.mutateAsync(request);
  };

  if (!identity) {
    return (
      <div className="container mx-auto px-4 py-12">
        <Card className="max-w-2xl mx-auto">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <AlertCircle className="h-5 w-5 text-primary" />
              Login Required
            </CardTitle>
            <CardDescription>
              Please login to access GST services
            </CardDescription>
          </CardHeader>
          <CardContent>
            <Button onClick={() => navigate({ to: '/' })}>
              Go to Home
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="container mx-auto px-4 py-8">
      <div className="max-w-3xl mx-auto">
        <div className="mb-8">
          <h1 className="text-3xl font-bold mb-2">💼 GST Services</h1>
          <p className="text-muted-foreground">
            GST registration and return filing services for your business
          </p>
        </div>

        <ServiceForm
          title="GST Application"
          description="Register for GST or file your GST returns"
          serviceType={ServiceType.gstServices}
          fee={3000}
          onSubmit={handleSubmit}
          fields={[
            { name: 'serviceType', label: 'Service Type', required: true, placeholder: 'GST Registration / GST Return Filing' },
            { name: 'businessName', label: 'Business Name', required: true, placeholder: 'Legal name of your business' },
            { name: 'businessType', label: 'Business Type', required: true, placeholder: 'Proprietorship / Partnership / Company' },
            { name: 'panNumber', label: 'PAN Number', required: true, placeholder: 'Business PAN number' },
            { name: 'turnover', label: 'Annual Turnover', required: true, placeholder: 'Expected/Actual annual turnover' },
            { name: 'businessAddress', label: 'Business Address', type: 'textarea', required: true, placeholder: 'Complete business address' },
          ]}
        />
      </div>
    </div>
  );
}
