import { useInternetIdentity } from '../hooks/useInternetIdentity';
import { useGetAllServiceRequests, useGetAllPaymentRecords, useUpdateRequestStatus, useVerifyPayment, useIsCallerAdmin } from '../hooks/useQueries';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { useNavigate } from '@tanstack/react-router';
import { Shield, FileText, CreditCard, AlertCircle, Loader2, Download } from 'lucide-react';
import { RequestStatus, PaymentStatus } from '../backend';
import { toast } from 'sonner';
import { useState } from 'react';

export default function AdminDashboard() {
  const { identity } = useInternetIdentity();
  const navigate = useNavigate();
  const { data: isAdmin, isLoading: adminCheckLoading } = useIsCallerAdmin();
  const { data: serviceRequests = [], isLoading: requestsLoading } = useGetAllServiceRequests();
  const { data: paymentRecords = [], isLoading: paymentsLoading } = useGetAllPaymentRecords();
  const updateStatus = useUpdateRequestStatus();
  const verifyPayment = useVerifyPayment();

  const [updatingRequest, setUpdatingRequest] = useState<string | null>(null);
  const [verifyingPayment, setVerifyingPayment] = useState<string | null>(null);

  if (!identity) {
    return (
      <div className="container mx-auto px-4 py-12">
        <Card className="max-w-2xl mx-auto">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <AlertCircle className="h-5 w-5 text-primary" />
              Login Required
            </CardTitle>
            <CardDescription>
              Please login to access the admin dashboard
            </CardDescription>
          </CardHeader>
          <CardContent>
            <Button onClick={() => navigate({ to: '/' })}>
              Go to Home
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  if (adminCheckLoading) {
    return (
      <div className="container mx-auto px-4 py-12 flex justify-center">
        <Loader2 className="h-8 w-8 animate-spin text-primary" />
      </div>
    );
  }

  if (!isAdmin) {
    return (
      <div className="container mx-auto px-4 py-12">
        <Card className="max-w-2xl mx-auto">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <AlertCircle className="h-5 w-5 text-destructive" />
              Access Denied
            </CardTitle>
            <CardDescription>
              You don't have permission to access the admin dashboard
            </CardDescription>
          </CardHeader>
          <CardContent>
            <Button onClick={() => navigate({ to: '/' })}>
              Go to Home
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  const handleStatusUpdate = async (requestId: string, status: RequestStatus) => {
    setUpdatingRequest(requestId);
    try {
      await updateStatus.mutateAsync({ requestId, status });
      toast.success('Request status updated successfully');
    } catch (error) {
      toast.error('Failed to update request status');
      console.error(error);
    } finally {
      setUpdatingRequest(null);
    }
  };

  const handlePaymentVerification = async (paymentId: string, status: PaymentStatus) => {
    setVerifyingPayment(paymentId);
    try {
      await verifyPayment.mutateAsync({ paymentId, status });
      toast.success('Payment status updated successfully');
    } catch (error) {
      toast.error('Failed to update payment status');
      console.error(error);
    } finally {
      setVerifyingPayment(null);
    }
  };

  const getStatusBadge = (status: RequestStatus) => {
    const variants: Record<RequestStatus, 'default' | 'secondary' | 'outline' | 'destructive'> = {
      [RequestStatus.submitted]: 'secondary',
      [RequestStatus.underReview]: 'default',
      [RequestStatus.approved]: 'outline',
      [RequestStatus.completed]: 'default',
    };
    return <Badge variant={variants[status]}>{status}</Badge>;
  };

  const getPaymentStatusBadge = (status: PaymentStatus) => {
    const variants: Record<PaymentStatus, 'default' | 'secondary' | 'outline' | 'destructive'> = {
      [PaymentStatus.pending]: 'secondary',
      [PaymentStatus.verified]: 'default',
      [PaymentStatus.rejected]: 'destructive',
    };
    return <Badge variant={variants[status]}>{status}</Badge>;
  };

  const formatDate = (timestamp: bigint) => {
    const date = new Date(Number(timestamp) / 1000000);
    return date.toLocaleDateString('en-IN', { year: 'numeric', month: 'short', day: 'numeric' });
  };

  const getServiceTypeName = (type: string) => {
    const names: Record<string, string> = {
      legalHelp: 'Legal Help',
      rtiServices: 'RTI Services',
      panIncomeTax: 'PAN & Income Tax',
      gstServices: 'GST Services',
      pfEsiServices: 'PF & ESI Services',
      societyTrustRegistration: 'Society/Trust Registration',
      accountingBookkeeping: 'Accounting & Bookkeeping',
    };
    return names[type] || type;
  };

  const downloadDocument = async (blob: any, filename: string) => {
    try {
      const bytes = await blob.getBytes();
      const blobObj = new Blob([bytes]);
      const url = URL.createObjectURL(blobObj);
      const a = document.createElement('a');
      a.href = url;
      a.download = filename;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
    } catch (error) {
      toast.error('Failed to download document');
      console.error(error);
    }
  };

  return (
    <div className="container mx-auto px-4 py-8">
      <div className="max-w-7xl mx-auto">
        <div className="mb-8">
          <h1 className="text-3xl font-bold mb-2 flex items-center gap-2">
            <Shield className="h-8 w-8 text-primary" />
            Admin Dashboard
          </h1>
          <p className="text-muted-foreground">
            Manage service requests and payment verifications
          </p>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2 text-lg">
                <FileText className="h-5 w-5" />
                Total Requests
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-primary">{serviceRequests.length}</div>
              <p className="text-sm text-muted-foreground">Service applications</p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2 text-lg">
                <CreditCard className="h-5 w-5" />
                Payments
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-primary">{paymentRecords.length}</div>
              <p className="text-sm text-muted-foreground">Payment records</p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2 text-lg">
                <Shield className="h-5 w-5" />
                Pending Review
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-primary">
                {serviceRequests.filter(r => r.status === RequestStatus.submitted).length}
              </div>
              <p className="text-sm text-muted-foreground">Awaiting review</p>
            </CardContent>
          </Card>
        </div>

        <Tabs defaultValue="requests" className="w-full">
          <TabsList className="grid w-full grid-cols-2">
            <TabsTrigger value="requests">Service Requests</TabsTrigger>
            <TabsTrigger value="payments">Payment Verification</TabsTrigger>
          </TabsList>

          <TabsContent value="requests" className="mt-6">
            <Card>
              <CardHeader>
                <CardTitle>All Service Requests</CardTitle>
                <CardDescription>Review and manage service applications</CardDescription>
              </CardHeader>
              <CardContent>
                {requestsLoading ? (
                  <div className="flex justify-center py-8">
                    <Loader2 className="h-6 w-6 animate-spin text-primary" />
                  </div>
                ) : serviceRequests.length === 0 ? (
                  <div className="text-center py-8 text-muted-foreground">
                    No service requests found
                  </div>
                ) : (
                  <div className="overflow-x-auto">
                    <Table>
                      <TableHeader>
                        <TableRow>
                          <TableHead>Request ID</TableHead>
                          <TableHead>Service Type</TableHead>
                          <TableHead>Date</TableHead>
                          <TableHead>Fee</TableHead>
                          <TableHead>Documents</TableHead>
                          <TableHead>Status</TableHead>
                          <TableHead>Action</TableHead>
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {serviceRequests.map((request) => (
                          <TableRow key={request.id}>
                            <TableCell className="font-mono text-xs">{request.id.slice(0, 12)}...</TableCell>
                            <TableCell className="font-medium">
                              {getServiceTypeName(request.serviceType)}
                            </TableCell>
                            <TableCell>{formatDate(request.createdAt)}</TableCell>
                            <TableCell>₹{Number(request.fee)}</TableCell>
                            <TableCell>
                              {request.documents.length > 0 && (
                                <Button
                                  variant="ghost"
                                  size="sm"
                                  onClick={() => downloadDocument(request.documents[0], `doc-${request.id}.pdf`)}
                                >
                                  <Download className="h-4 w-4 mr-1" />
                                  {request.documents.length}
                                </Button>
                              )}
                            </TableCell>
                            <TableCell>{getStatusBadge(request.status)}</TableCell>
                            <TableCell>
                              <Select
                                value={request.status}
                                onValueChange={(value) => handleStatusUpdate(request.id, value as RequestStatus)}
                                disabled={updatingRequest === request.id}
                              >
                                <SelectTrigger className="w-[140px]">
                                  <SelectValue />
                                </SelectTrigger>
                                <SelectContent>
                                  <SelectItem value={RequestStatus.submitted}>Submitted</SelectItem>
                                  <SelectItem value={RequestStatus.underReview}>Under Review</SelectItem>
                                  <SelectItem value={RequestStatus.approved}>Approved</SelectItem>
                                  <SelectItem value={RequestStatus.completed}>Completed</SelectItem>
                                </SelectContent>
                              </Select>
                            </TableCell>
                          </TableRow>
                        ))}
                      </TableBody>
                    </Table>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="payments" className="mt-6">
            <Card>
              <CardHeader>
                <CardTitle>Payment Verification</CardTitle>
                <CardDescription>Verify and manage payment records</CardDescription>
              </CardHeader>
              <CardContent>
                {paymentsLoading ? (
                  <div className="flex justify-center py-8">
                    <Loader2 className="h-6 w-6 animate-spin text-primary" />
                  </div>
                ) : paymentRecords.length === 0 ? (
                  <div className="text-center py-8 text-muted-foreground">
                    No payment records found
                  </div>
                ) : (
                  <div className="overflow-x-auto">
                    <Table>
                      <TableHeader>
                        <TableRow>
                          <TableHead>Payment ID</TableHead>
                          <TableHead>Date</TableHead>
                          <TableHead>Amount</TableHead>
                          <TableHead>Proof</TableHead>
                          <TableHead>Status</TableHead>
                          <TableHead>Action</TableHead>
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {paymentRecords.map((record) => (
                          <TableRow key={record.id}>
                            <TableCell className="font-mono text-xs">{record.id.slice(0, 12)}...</TableCell>
                            <TableCell>{formatDate(record.createdAt)}</TableCell>
                            <TableCell>₹{Number(record.amount)}</TableCell>
                            <TableCell>
                              <Button
                                variant="ghost"
                                size="sm"
                                onClick={() => downloadDocument(record.paymentProof, `payment-${record.id}.jpg`)}
                              >
                                <Download className="h-4 w-4 mr-1" />
                                View
                              </Button>
                            </TableCell>
                            <TableCell>{getPaymentStatusBadge(record.status)}</TableCell>
                            <TableCell>
                              <Select
                                value={record.status}
                                onValueChange={(value) => handlePaymentVerification(record.id, value as PaymentStatus)}
                                disabled={verifyingPayment === record.id}
                              >
                                <SelectTrigger className="w-[120px]">
                                  <SelectValue />
                                </SelectTrigger>
                                <SelectContent>
                                  <SelectItem value={PaymentStatus.pending}>Pending</SelectItem>
                                  <SelectItem value={PaymentStatus.verified}>Verified</SelectItem>
                                  <SelectItem value={PaymentStatus.rejected}>Rejected</SelectItem>
                                </SelectContent>
                              </Select>
                            </TableCell>
                          </TableRow>
                        ))}
                      </TableBody>
                    </Table>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  );
}
