import { useInternetIdentity } from '../hooks/useInternetIdentity';
import { useSubmitServiceRequest } from '../hooks/useQueries';
import { ServiceType, RequestStatus, ExternalBlob } from '../backend';
import ServiceForm from '../components/ServiceForm';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { useNavigate } from '@tanstack/react-router';
import { AlertCircle } from 'lucide-react';

export default function AccountingPage() {
  const { identity } = useInternetIdentity();
  const navigate = useNavigate();
  const submitRequest = useSubmitServiceRequest();

  const handleSubmit = async (details: string, documents: ExternalBlob[]) => {
    if (!identity) return;

    const request = {
      id: `accounting-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
      user: identity.getPrincipal(),
      serviceType: ServiceType.accountingBookkeeping,
      details,
      documents,
      status: RequestStatus.submitted,
      fee: BigInt(4000),
      createdAt: BigInt(Date.now() * 1000000),
    };

    await submitRequest.mutateAsync(request);
  };

  if (!identity) {
    return (
      <div className="container mx-auto px-4 py-12">
        <Card className="max-w-2xl mx-auto">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <AlertCircle className="h-5 w-5 text-primary" />
              Login Required
            </CardTitle>
            <CardDescription>
              Please login to access accounting services
            </CardDescription>
          </CardHeader>
          <CardContent>
            <Button onClick={() => navigate({ to: '/' })}>
              Go to Home
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="container mx-auto px-4 py-8">
      <div className="max-w-3xl mx-auto">
        <div className="mb-8">
          <h1 className="text-3xl font-bold mb-2">📚 Accounting & Bookkeeping</h1>
          <p className="text-muted-foreground">
            Professional accounting, bookkeeping, and audit services for your business
          </p>
        </div>

        <ServiceForm
          title="Accounting Services Request"
          description="Request accounting and bookkeeping services"
          serviceType={ServiceType.accountingBookkeeping}
          fee={4000}
          onSubmit={handleSubmit}
          fields={[
            { name: 'serviceType', label: 'Service Type', required: true, placeholder: 'Bookkeeping / Accounting / Audit / Tax Planning' },
            { name: 'businessName', label: 'Business Name', required: true, placeholder: 'Name of your business' },
            { name: 'businessNature', label: 'Nature of Business', required: true, placeholder: 'Type of business activity' },
            { name: 'financialYear', label: 'Financial Year', required: true, placeholder: 'e.g., 2024-25' },
            { name: 'transactionVolume', label: 'Monthly Transaction Volume', required: true, placeholder: 'Approximate number of transactions' },
            { name: 'requirements', label: 'Specific Requirements', type: 'textarea', required: true, placeholder: 'Describe your accounting needs' },
          ]}
        />
      </div>
    </div>
  );
}
