import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import { useActor } from './useActor';
import { UserProfile, ServiceRequest, PaymentRecord, RequestStatus, PaymentStatus, ExternalBlob } from '../backend';
import { Principal } from '@icp-sdk/core/principal';

export function useGetCallerUserProfile() {
  const { actor, isFetching: actorFetching } = useActor();

  const query = useQuery<UserProfile | null>({
    queryKey: ['currentUserProfile'],
    queryFn: async () => {
      if (!actor) throw new Error('Actor not available');
      return actor.getCallerUserProfile();
    },
    enabled: !!actor && !actorFetching,
    retry: false,
  });

  return {
    ...query,
    isLoading: actorFetching || query.isLoading,
    isFetched: !!actor && query.isFetched,
  };
}

export function useSaveCallerUserProfile() {
  const { actor } = useActor();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: async (profile: UserProfile) => {
      if (!actor) throw new Error('Actor not available');
      return actor.saveCallerUserProfile(profile);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['currentUserProfile'] });
    },
  });
}

export function useIsCallerAdmin() {
  const { actor, isFetching } = useActor();

  return useQuery<boolean>({
    queryKey: ['isAdmin'],
    queryFn: async () => {
      if (!actor) return false;
      return actor.isCallerAdmin();
    },
    enabled: !!actor && !isFetching,
  });
}

export function useGenerateOtp() {
  const { actor } = useActor();

  return useMutation({
    mutationFn: async (identifier: string) => {
      if (!actor) throw new Error('Actor not available');
      return actor.generateOtp(identifier);
    },
  });
}

export function useVerifyOtp() {
  const { actor } = useActor();

  return useMutation({
    mutationFn: async ({ identifier, otp }: { identifier: string; otp: number }) => {
      if (!actor) throw new Error('Actor not available');
      return actor.verifyOtp(identifier, BigInt(otp));
    },
  });
}

export function useSubmitServiceRequest() {
  const { actor } = useActor();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: async (request: ServiceRequest) => {
      if (!actor) throw new Error('Actor not available');
      return actor.submitServiceRequest(request);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['userServiceRequests'] });
      queryClient.invalidateQueries({ queryKey: ['currentUserProfile'] });
    },
  });
}

export function useUploadPaymentProof() {
  const { actor } = useActor();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: async (record: PaymentRecord) => {
      if (!actor) throw new Error('Actor not available');
      return actor.uploadPaymentProof(record);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['userPaymentRecords'] });
    },
  });
}

export function useGetUserServiceRequests() {
  const { actor, isFetching } = useActor();

  return useQuery<ServiceRequest[]>({
    queryKey: ['userServiceRequests'],
    queryFn: async () => {
      if (!actor) return [];
      return actor.getUserServiceRequests();
    },
    enabled: !!actor && !isFetching,
  });
}

export function useGetUserPaymentRecords() {
  const { actor, isFetching } = useActor();

  return useQuery<PaymentRecord[]>({
    queryKey: ['userPaymentRecords'],
    queryFn: async () => {
      if (!actor) return [];
      return actor.getUserPaymentRecords();
    },
    enabled: !!actor && !isFetching,
  });
}

export function useGetAllServiceRequests() {
  const { actor, isFetching } = useActor();

  return useQuery<ServiceRequest[]>({
    queryKey: ['allServiceRequests'],
    queryFn: async () => {
      if (!actor) return [];
      return actor.getAllServiceRequests();
    },
    enabled: !!actor && !isFetching,
  });
}

export function useGetAllPaymentRecords() {
  const { actor, isFetching } = useActor();

  return useQuery<PaymentRecord[]>({
    queryKey: ['allPaymentRecords'],
    queryFn: async () => {
      if (!actor) return [];
      return actor.getAllPaymentRecords();
    },
    enabled: !!actor && !isFetching,
  });
}

export function useUpdateRequestStatus() {
  const { actor } = useActor();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: async ({ requestId, status }: { requestId: string; status: RequestStatus }) => {
      if (!actor) throw new Error('Actor not available');
      return actor.updateRequestStatus(requestId, status);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['allServiceRequests'] });
    },
  });
}

export function useVerifyPayment() {
  const { actor } = useActor();
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: async ({ paymentId, status }: { paymentId: string; status: PaymentStatus }) => {
      if (!actor) throw new Error('Actor not available');
      return actor.verifyPayment(paymentId, status);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['allPaymentRecords'] });
    },
  });
}

export function useGetUpiQrCode() {
  const { actor, isFetching } = useActor();

  return useQuery<ExternalBlob | null>({
    queryKey: ['upiQrCode'],
    queryFn: async () => {
      if (!actor) return null;
      return actor.getUpiQrCode();
    },
    enabled: !!actor && !isFetching,
  });
}
