import { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Upload, X, Loader2 } from 'lucide-react';
import { ExternalBlob, ServiceType } from '../backend';
import { toast } from 'sonner';

interface ServiceFormProps {
  title: string;
  description: string;
  serviceType: ServiceType;
  fee: number;
  onSubmit: (details: string, documents: ExternalBlob[]) => Promise<void>;
  fields?: Array<{ name: string; label: string; type?: string; required?: boolean; placeholder?: string }>;
}

export default function ServiceForm({ title, description, serviceType, fee, onSubmit, fields = [] }: ServiceFormProps) {
  const [formData, setFormData] = useState<Record<string, string>>({});
  const [files, setFiles] = useState<File[]>([]);
  const [uploading, setUploading] = useState(false);
  const [uploadProgress, setUploadProgress] = useState<Record<string, number>>({});

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files) {
      const newFiles = Array.from(e.target.files);
      const validFiles = newFiles.filter(file => {
        const ext = file.name.split('.').pop()?.toLowerCase();
        return ['pdf', 'jpg', 'jpeg', 'png', 'doc', 'docx'].includes(ext || '');
      });
      
      if (validFiles.length !== newFiles.length) {
        toast.error('Some files were skipped. Only PDF, JPG, PNG, DOC files are allowed.');
      }
      
      setFiles(prev => [...prev, ...validFiles]);
    }
  };

  const removeFile = (index: number) => {
    setFiles(prev => prev.filter((_, i) => i !== index));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    // Validate required fields
    const missingFields = fields.filter(field => field.required && !formData[field.name]?.trim());
    if (missingFields.length > 0) {
      toast.error('Please fill in all required fields');
      return;
    }

    setUploading(true);
    
    try {
      // Convert files to ExternalBlob
      const documentBlobs: ExternalBlob[] = [];
      
      for (let i = 0; i < files.length; i++) {
        const file = files[i];
        const arrayBuffer = await file.arrayBuffer();
        const uint8Array = new Uint8Array(arrayBuffer);
        
        const blob = ExternalBlob.fromBytes(uint8Array).withUploadProgress((percentage) => {
          setUploadProgress(prev => ({ ...prev, [file.name]: percentage }));
        });
        
        documentBlobs.push(blob);
      }

      // Create details string from form data
      const details = JSON.stringify(formData);
      
      await onSubmit(details, documentBlobs);
      
      // Reset form
      setFormData({});
      setFiles([]);
      setUploadProgress({});
      toast.success('Application submitted successfully!');
    } catch (error) {
      console.error('Submission error:', error);
      toast.error('Failed to submit application');
    } finally {
      setUploading(false);
    }
  };

  return (
    <Card>
      <CardHeader>
        <CardTitle>{title}</CardTitle>
        <CardDescription>{description}</CardDescription>
        <div className="mt-2">
          <span className="text-sm font-medium">Service Fee: </span>
          <span className="text-lg font-bold text-primary">₹{fee}</span>
        </div>
      </CardHeader>
      <CardContent>
        <form onSubmit={handleSubmit} className="space-y-6">
          {fields.map((field) => (
            <div key={field.name} className="space-y-2">
              <Label htmlFor={field.name}>
                {field.label} {field.required && <span className="text-destructive">*</span>}
              </Label>
              {field.type === 'textarea' ? (
                <Textarea
                  id={field.name}
                  value={formData[field.name] || ''}
                  onChange={(e) => setFormData(prev => ({ ...prev, [field.name]: e.target.value }))}
                  placeholder={field.placeholder}
                  required={field.required}
                  rows={4}
                />
              ) : (
                <Input
                  id={field.name}
                  type={field.type || 'text'}
                  value={formData[field.name] || ''}
                  onChange={(e) => setFormData(prev => ({ ...prev, [field.name]: e.target.value }))}
                  placeholder={field.placeholder}
                  required={field.required}
                />
              )}
            </div>
          ))}

          <div className="space-y-2">
            <Label htmlFor="documents">Upload Documents (PDF, JPG, DOC)</Label>
            <div className="flex items-center gap-2">
              <Input
                id="documents"
                type="file"
                onChange={handleFileChange}
                multiple
                accept=".pdf,.jpg,.jpeg,.png,.doc,.docx"
                className="cursor-pointer"
              />
              <Upload className="h-5 w-5 text-muted-foreground" />
            </div>
            {files.length > 0 && (
              <div className="mt-2 space-y-2">
                {files.map((file, index) => (
                  <div key={index} className="flex items-center justify-between p-2 bg-muted rounded-md">
                    <span className="text-sm truncate flex-1">{file.name}</span>
                    {uploadProgress[file.name] !== undefined && uploadProgress[file.name] < 100 && (
                      <span className="text-xs text-muted-foreground mx-2">
                        {uploadProgress[file.name]}%
                      </span>
                    )}
                    <Button
                      type="button"
                      variant="ghost"
                      size="icon"
                      onClick={() => removeFile(index)}
                      disabled={uploading}
                    >
                      <X className="h-4 w-4" />
                    </Button>
                  </div>
                ))}
              </div>
            )}
          </div>

          <Button type="submit" className="w-full" disabled={uploading}>
            {uploading ? (
              <>
                <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                Submitting...
              </>
            ) : (
              'Submit Application'
            )}
          </Button>
        </form>
      </CardContent>
    </Card>
  );
}
