import { useState } from 'react';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { InputOTP, InputOTPGroup, InputOTPSlot } from '@/components/ui/input-otp';
import { toast } from 'sonner';
import { useGenerateOtp, useVerifyOtp } from '../hooks/useQueries';
import { Loader2, Mail, Phone } from 'lucide-react';
import { Alert, AlertDescription } from '@/components/ui/alert';

interface OtpLoginDialogProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  onSuccess: () => void;
}

export default function OtpLoginDialog({ open, onOpenChange, onSuccess }: OtpLoginDialogProps) {
  const [step, setStep] = useState<'identifier' | 'otp'>('identifier');
  const [identifier, setIdentifier] = useState('');
  const [otp, setOtp] = useState('');
  const [generatedOtp, setGeneratedOtp] = useState<string>('');
  const [identifierType, setIdentifierType] = useState<'email' | 'mobile'>('mobile');

  const generateOtp = useGenerateOtp();
  const verifyOtp = useVerifyOtp();

  const handleIdentifierSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!identifier.trim()) {
      toast.error('Please enter your mobile number or email');
      return;
    }

    // Validate identifier format
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    const mobileRegex = /^[6-9]\d{9}$/;
    
    if (!emailRegex.test(identifier) && !mobileRegex.test(identifier)) {
      toast.error('Please enter a valid mobile number (10 digits) or email address');
      return;
    }

    try {
      const otpValue = await generateOtp.mutateAsync(identifier.trim());
      setGeneratedOtp(otpValue.toString());
      setStep('otp');
      toast.success('OTP generated successfully!');
    } catch (error) {
      toast.error('Failed to generate OTP. Please try again.');
      console.error(error);
    }
  };

  const handleOtpSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (otp.length !== 6) {
      toast.error('Please enter the complete 6-digit OTP');
      return;
    }

    try {
      const isValid = await verifyOtp.mutateAsync({
        identifier: identifier.trim(),
        otp: parseInt(otp),
      });

      if (isValid) {
        toast.success('Login successful!');
        onSuccess();
        handleClose();
      } else {
        toast.error('Invalid or expired OTP. Please try again.');
        setOtp('');
      }
    } catch (error) {
      toast.error('Verification failed. Please try again.');
      console.error(error);
    }
  };

  const handleClose = () => {
    setStep('identifier');
    setIdentifier('');
    setOtp('');
    setGeneratedOtp('');
    onOpenChange(false);
  };

  const handleBack = () => {
    setStep('identifier');
    setOtp('');
    setGeneratedOtp('');
  };

  return (
    <Dialog open={open} onOpenChange={handleClose}>
      <DialogContent className="sm:max-w-md">
        <DialogHeader>
          <DialogTitle>
            {step === 'identifier' ? 'Login / Register' : 'Verify OTP'}
          </DialogTitle>
          <DialogDescription>
            {step === 'identifier' 
              ? 'Enter your mobile number or email to receive an OTP'
              : 'Enter the 6-digit OTP to continue'
            }
          </DialogDescription>
        </DialogHeader>

        {step === 'identifier' ? (
          <form onSubmit={handleIdentifierSubmit} className="space-y-4">
            <div className="space-y-2">
              <Label htmlFor="identifier">Mobile Number or Email *</Label>
              <div className="relative">
                <Input
                  id="identifier"
                  value={identifier}
                  onChange={(e) => {
                    const value = e.target.value;
                    setIdentifier(value);
                    // Auto-detect type
                    if (value.includes('@')) {
                      setIdentifierType('email');
                    } else {
                      setIdentifierType('mobile');
                    }
                  }}
                  placeholder="Enter mobile number or email"
                  className="pl-10"
                  required
                />
                <div className="absolute left-3 top-1/2 -translate-y-1/2 text-muted-foreground">
                  {identifierType === 'email' ? (
                    <Mail className="h-4 w-4" />
                  ) : (
                    <Phone className="h-4 w-4" />
                  )}
                </div>
              </div>
              <p className="text-xs text-muted-foreground">
                Enter a 10-digit mobile number or valid email address
              </p>
            </div>
            <Button 
              type="submit" 
              className="w-full" 
              disabled={generateOtp.isPending}
            >
              {generateOtp.isPending ? (
                <>
                  <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                  Generating OTP...
                </>
              ) : (
                'Generate OTP'
              )}
            </Button>
          </form>
        ) : (
          <form onSubmit={handleOtpSubmit} className="space-y-4">
            {generatedOtp && (
              <Alert className="bg-primary/10 border-primary/20">
                <AlertDescription className="text-center">
                  <div className="font-semibold text-sm mb-1">Demo OTP (for testing):</div>
                  <div className="text-2xl font-bold tracking-wider text-primary">
                    {generatedOtp}
                  </div>
                  <div className="text-xs text-muted-foreground mt-1">
                    In production, this would be sent via SMS/Email
                  </div>
                </AlertDescription>
              </Alert>
            )}

            <div className="space-y-2">
              <Label htmlFor="otp">Enter OTP *</Label>
              <div className="flex justify-center">
                <InputOTP
                  maxLength={6}
                  value={otp}
                  onChange={(value) => setOtp(value)}
                >
                  <InputOTPGroup>
                    <InputOTPSlot index={0} />
                    <InputOTPSlot index={1} />
                    <InputOTPSlot index={2} />
                    <InputOTPSlot index={3} />
                    <InputOTPSlot index={4} />
                    <InputOTPSlot index={5} />
                  </InputOTPGroup>
                </InputOTP>
              </div>
              <p className="text-xs text-muted-foreground text-center">
                OTP sent to {identifier}
              </p>
            </div>

            <div className="flex gap-2">
              <Button 
                type="button" 
                variant="outline" 
                className="flex-1"
                onClick={handleBack}
                disabled={verifyOtp.isPending}
              >
                Back
              </Button>
              <Button 
                type="submit" 
                className="flex-1" 
                disabled={verifyOtp.isPending || otp.length !== 6}
              >
                {verifyOtp.isPending ? (
                  <>
                    <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                    Verifying...
                  </>
                ) : (
                  'Verify & Login'
                )}
              </Button>
            </div>

            <div className="text-center">
              <Button
                type="button"
                variant="link"
                size="sm"
                onClick={handleIdentifierSubmit}
                disabled={generateOtp.isPending}
                className="text-xs"
              >
                Resend OTP
              </Button>
            </div>
          </form>
        )}
      </DialogContent>
    </Dialog>
  );
}
