import { Outlet, useNavigate, useLocation } from '@tanstack/react-router';
import { useInternetIdentity } from '../hooks/useInternetIdentity';
import { useGetCallerUserProfile, useIsCallerAdmin } from '../hooks/useQueries';
import { Button } from '@/components/ui/button';
import { Sheet, SheetContent, SheetTrigger } from '@/components/ui/sheet';
import { Menu, User, LogOut, Shield, LogIn } from 'lucide-react';
import { useState } from 'react';
import { useQueryClient } from '@tanstack/react-query';
import ProfileSetupDialog from './ProfileSetupDialog';
import OtpLoginDialog from './OtpLoginDialog';

export default function Layout() {
  const navigate = useNavigate();
  const location = useLocation();
  const { identity, clear, login } = useInternetIdentity();
  const queryClient = useQueryClient();
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);
  const [otpDialogOpen, setOtpDialogOpen] = useState(false);

  const isAuthenticated = !!identity;
  const { data: userProfile, isLoading: profileLoading, isFetched } = useGetCallerUserProfile();
  const { data: isAdmin } = useIsCallerAdmin();

  const showProfileSetup = isAuthenticated && !profileLoading && isFetched && userProfile === null;

  const handleLogout = async () => {
    await clear();
    queryClient.clear();
    navigate({ to: '/' });
  };

  const handleLoginClick = () => {
    setOtpDialogOpen(true);
  };

  const handleOtpSuccess = async () => {
    // After OTP verification, log in with Internet Identity
    try {
      await login();
    } catch (error: any) {
      console.error('Login error:', error);
      if (error.message === 'User is already authenticated') {
        await clear();
        setTimeout(() => login(), 300);
      }
    }
  };

  const navItems = [
    { path: '/', label: 'Home', icon: '🏠' },
    { path: '/legal-help', label: 'Legal Help', icon: '⚖️' },
    { path: '/rti-services', label: 'RTI Services', icon: '🧾' },
    { path: '/pan-tax', label: 'PAN & Tax', icon: '🧍‍♂️' },
    { path: '/gst-services', label: 'GST Services', icon: '💼' },
    { path: '/pf-esi', label: 'PF & ESI', icon: '🏢' },
    { path: '/society-trust', label: 'Society/Trust', icon: '🏛️' },
    { path: '/accounting', label: 'Accounting', icon: '📚' },
    { path: '/payment', label: 'Payment', icon: '💳' },
    { path: '/contact', label: 'Contact', icon: '📞' },
  ];

  return (
    <div className="min-h-screen flex flex-col bg-gradient-to-br from-background via-background to-muted/20">
      <header className="sticky top-0 z-50 w-full border-b bg-background/95 backdrop-blur supports-[backdrop-filter]:bg-background/60">
        <div className="container mx-auto px-4">
          <div className="flex h-20 items-center justify-between">
            <div className="flex items-center gap-3 cursor-pointer" onClick={() => navigate({ to: '/' })}>
              <img 
                src="/assets/IMG-20250826-WA0003.jpg" 
                alt="Juristroid Logo" 
                className="h-14 w-14 object-contain" 
              />
              <div className="flex flex-col">
                <span className="text-2xl font-bold bg-gradient-to-r from-[oklch(0.65_0.20_250)] to-[oklch(0.55_0.18_280)] bg-clip-text text-transparent leading-tight brand-name">
                  Juristroid
                </span>
                <span className="text-xs text-muted-foreground font-medium hindi-tagline">
                  आपका अपना वकील
                </span>
              </div>
            </div>

            <nav className="hidden md:flex items-center gap-1">
              {navItems.map((item) => (
                <Button
                  key={item.path}
                  variant={location.pathname === item.path ? 'default' : 'ghost'}
                  size="sm"
                  onClick={() => navigate({ to: item.path })}
                  className="gap-2"
                >
                  <span>{item.icon}</span>
                  <span className="text-sm">{item.label}</span>
                </Button>
              ))}
            </nav>

            <div className="flex items-center gap-2">
              {isAuthenticated ? (
                <>
                  {isAdmin && (
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => navigate({ to: '/admin' })}
                      className="hidden md:flex gap-2"
                    >
                      <Shield className="h-4 w-4" />
                      Admin
                    </Button>
                  )}
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => navigate({ to: '/profile' })}
                    className="hidden md:flex gap-2"
                  >
                    <User className="h-4 w-4" />
                    Profile
                  </Button>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={handleLogout}
                    className="hidden md:flex gap-2"
                  >
                    <LogOut className="h-4 w-4" />
                    Logout
                  </Button>
                </>
              ) : (
                <Button
                  onClick={handleLoginClick}
                  size="sm"
                  className="hidden md:flex gap-2"
                >
                  <LogIn className="h-4 w-4" />
                  Login
                </Button>
              )}

              <Sheet open={mobileMenuOpen} onOpenChange={setMobileMenuOpen}>
                <SheetTrigger asChild>
                  <Button variant="ghost" size="icon" className="md:hidden">
                    <Menu className="h-5 w-5" />
                  </Button>
                </SheetTrigger>
                <SheetContent side="right" className="w-[280px]">
                  <div className="flex flex-col gap-4 mt-8">
                    {navItems.map((item) => (
                      <Button
                        key={item.path}
                        variant={location.pathname === item.path ? 'default' : 'ghost'}
                        onClick={() => {
                          navigate({ to: item.path });
                          setMobileMenuOpen(false);
                        }}
                        className="justify-start gap-2"
                      >
                        <span>{item.icon}</span>
                        {item.label}
                      </Button>
                    ))}
                    {isAuthenticated ? (
                      <>
                        {isAdmin && (
                          <Button
                            variant="outline"
                            onClick={() => {
                              navigate({ to: '/admin' });
                              setMobileMenuOpen(false);
                            }}
                            className="justify-start gap-2"
                          >
                            <Shield className="h-4 w-4" />
                            Admin Dashboard
                          </Button>
                        )}
                        <Button
                          variant="outline"
                          onClick={() => {
                            navigate({ to: '/profile' });
                            setMobileMenuOpen(false);
                          }}
                          className="justify-start gap-2"
                        >
                          <User className="h-4 w-4" />
                          My Profile
                        </Button>
                        <Button
                          variant="ghost"
                          onClick={handleLogout}
                          className="justify-start gap-2"
                        >
                          <LogOut className="h-4 w-4" />
                          Logout
                        </Button>
                      </>
                    ) : (
                      <Button onClick={handleLoginClick} className="justify-start gap-2">
                        <LogIn className="h-4 w-4" />
                        Login
                      </Button>
                    )}
                  </div>
                </SheetContent>
              </Sheet>
            </div>
          </div>
        </div>
      </header>

      <main className="flex-1">
        <Outlet />
      </main>

      <footer className="border-t bg-muted/30 py-8 mt-12">
        <div className="container mx-auto px-4">
          <div className="text-center text-sm text-muted-foreground">
            © 2025. Built with love using{' '}
            <a href="https://caffeine.ai" target="_blank" rel="noopener noreferrer" className="text-primary hover:underline">
              caffeine.ai
            </a>
          </div>
        </div>
      </footer>

      {showProfileSetup && <ProfileSetupDialog />}
      <OtpLoginDialog 
        open={otpDialogOpen} 
        onOpenChange={setOtpDialogOpen}
        onSuccess={handleOtpSuccess}
      />
    </div>
  );
}
