import AccessControl "authorization/access-control";
import Principal "mo:base/Principal";
import OrderedMap "mo:base/OrderedMap";
import Debug "mo:base/Debug";
import Storage "blob-storage/Storage";
import MixinStorage "blob-storage/Mixin";
import Time "mo:base/Time";
import Text "mo:base/Text";
import List "mo:base/List";
import Iter "mo:base/Iter";
import Nat "mo:base/Nat";
import Int "mo:base/Int";

actor Juristroid {
  let accessControlState = AccessControl.initState();
  let storage = Storage.new();
  include MixinStorage(storage);

  transient let principalMap = OrderedMap.Make<Principal>(Principal.compare);
  transient let textMap = OrderedMap.Make<Text>(Text.compare);

  var userProfiles = principalMap.empty<UserProfile>();
  var serviceRequests = textMap.empty<ServiceRequest>();
  var paymentRecords = textMap.empty<PaymentRecord>();
  var upiQrCode : ?Storage.ExternalBlob = null;
  var otpRecords = textMap.empty<OtpRecord>();

  public shared ({ caller }) func initializeAccessControl() : async () {
    AccessControl.initialize(accessControlState, caller);
  };

  public query ({ caller }) func getCallerUserRole() : async AccessControl.UserRole {
    AccessControl.getUserRole(accessControlState, caller);
  };

  public shared ({ caller }) func assignCallerUserRole(user : Principal, role : AccessControl.UserRole) : async () {
    AccessControl.assignRole(accessControlState, caller, user, role);
  };

  public query ({ caller }) func isCallerAdmin() : async Bool {
    AccessControl.isAdmin(accessControlState, caller);
  };

  public type UserProfile = {
    name : Text;
    contact : Text;
    email : Text;
    applications : [Text];
  };

  public query ({ caller }) func getCallerUserProfile() : async ?UserProfile {
    if (not (AccessControl.hasPermission(accessControlState, caller, #user))) {
      Debug.trap("Unauthorized: Only users can access profiles");
    };
    principalMap.get(userProfiles, caller);
  };

  public query ({ caller }) func getUserProfile(user : Principal) : async ?UserProfile {
    if (caller != user and not AccessControl.isAdmin(accessControlState, caller)) {
      Debug.trap("Unauthorized: Can only view your own profile");
    };
    principalMap.get(userProfiles, user);
  };

  public shared ({ caller }) func saveCallerUserProfile(profile : UserProfile) : async () {
    if (not (AccessControl.hasPermission(accessControlState, caller, #user))) {
      Debug.trap("Unauthorized: Only users can save profiles");
    };
    userProfiles := principalMap.put(userProfiles, caller, profile);
  };

  public type ServiceRequest = {
    id : Text;
    user : Principal;
    serviceType : ServiceType;
    details : Text;
    documents : [Storage.ExternalBlob];
    status : RequestStatus;
    fee : Nat;
    createdAt : Time.Time;
  };

  public type PaymentRecord = {
    id : Text;
    user : Principal;
    amount : Nat;
    paymentProof : Storage.ExternalBlob;
    status : PaymentStatus;
    createdAt : Time.Time;
  };

  public type ServiceType = {
    #legalHelp;
    #rtiServices;
    #panIncomeTax;
    #gstServices;
    #pfEsiServices;
    #societyTrustRegistration;
    #accountingBookkeeping;
  };

  public type RequestStatus = {
    #submitted;
    #underReview;
    #approved;
    #completed;
  };

  public type PaymentStatus = {
    #pending;
    #verified;
    #rejected;
  };

  public type OtpRecord = {
    identifier : Text;
    otp : Nat;
    createdAt : Time.Time;
    verified : Bool;
  };

  public shared func generateOtp(identifier : Text) : async Nat {
    let otp = Int.abs(Int.rem(Time.now(), 1000000));
    let record : OtpRecord = {
      identifier;
      otp;
      createdAt = Time.now();
      verified = false;
    };
    otpRecords := textMap.put(otpRecords, identifier, record);
    otp;
  };

  public shared func verifyOtp(identifier : Text, otp : Nat) : async Bool {
    switch (textMap.get(otpRecords, identifier)) {
      case (null) { false };
      case (?record) {
        if (record.otp == otp and Time.now() - record.createdAt < 300_000_000_000) {
          let updatedRecord = {
            identifier = record.identifier;
            otp = record.otp;
            createdAt = record.createdAt;
            verified = true;
          };
          otpRecords := textMap.put(otpRecords, identifier, updatedRecord);
          true;
        } else {
          false;
        };
      };
    };
  };

  public shared ({ caller }) func submitServiceRequest(request : ServiceRequest) : async () {
    if (not (AccessControl.hasPermission(accessControlState, caller, #user))) {
      Debug.trap("Unauthorized: Only users can submit requests");
    };
    if (request.user != caller) {
      Debug.trap("Unauthorized: Cannot submit requests for other users");
    };
    serviceRequests := textMap.put(serviceRequests, request.id, request);
  };

  public shared ({ caller }) func uploadPaymentProof(record : PaymentRecord) : async () {
    if (not (AccessControl.hasPermission(accessControlState, caller, #user))) {
      Debug.trap("Unauthorized: Only users can upload payment proof");
    };
    if (record.user != caller) {
      Debug.trap("Unauthorized: Cannot upload payment proof for other users");
    };
    paymentRecords := textMap.put(paymentRecords, record.id, record);
  };

  public query ({ caller }) func getUserServiceRequests() : async [ServiceRequest] {
    if (not (AccessControl.hasPermission(accessControlState, caller, #user))) {
      Debug.trap("Unauthorized: Only users can view requests");
    };
    var userRequests = List.nil<ServiceRequest>();
    for ((_, request) in textMap.entries(serviceRequests)) {
      if (request.user == caller) {
        userRequests := List.push(request, userRequests);
      };
    };
    List.toArray(userRequests);
  };

  public query ({ caller }) func getUserPaymentRecords() : async [PaymentRecord] {
    if (not (AccessControl.hasPermission(accessControlState, caller, #user))) {
      Debug.trap("Unauthorized: Only users can view payment records");
    };
    var userPayments = List.nil<PaymentRecord>();
    for ((_, record) in textMap.entries(paymentRecords)) {
      if (record.user == caller) {
        userPayments := List.push(record, userPayments);
      };
    };
    List.toArray(userPayments);
  };

  public query ({ caller }) func getAllServiceRequests() : async [ServiceRequest] {
    if (not (AccessControl.hasPermission(accessControlState, caller, #admin))) {
      Debug.trap("Unauthorized: Only admins can view all requests");
    };
    Iter.toArray(textMap.vals(serviceRequests));
  };

  public query ({ caller }) func getAllPaymentRecords() : async [PaymentRecord] {
    if (not (AccessControl.hasPermission(accessControlState, caller, #admin))) {
      Debug.trap("Unauthorized: Only admins can view all payment records");
    };
    Iter.toArray(textMap.vals(paymentRecords));
  };

  public shared ({ caller }) func updateRequestStatus(requestId : Text, status : RequestStatus) : async () {
    if (not (AccessControl.hasPermission(accessControlState, caller, #admin))) {
      Debug.trap("Unauthorized: Only admins can update request status");
    };
    switch (textMap.get(serviceRequests, requestId)) {
      case (null) { Debug.trap("Request not found") };
      case (?request) {
        let updatedRequest = {
          id = request.id;
          user = request.user;
          serviceType = request.serviceType;
          details = request.details;
          documents = request.documents;
          status;
          fee = request.fee;
          createdAt = request.createdAt;
        };
        serviceRequests := textMap.put(serviceRequests, requestId, updatedRequest);
      };
    };
  };

  public shared ({ caller }) func verifyPayment(paymentId : Text, status : PaymentStatus) : async () {
    if (not (AccessControl.hasPermission(accessControlState, caller, #admin))) {
      Debug.trap("Unauthorized: Only admins can verify payments");
    };
    switch (textMap.get(paymentRecords, paymentId)) {
      case (null) { Debug.trap("Payment record not found") };
      case (?record) {
        let updatedRecord = {
          id = record.id;
          user = record.user;
          amount = record.amount;
          paymentProof = record.paymentProof;
          status;
          createdAt = record.createdAt;
        };
        paymentRecords := textMap.put(paymentRecords, paymentId, updatedRecord);
      };
    };
  };

  public shared ({ caller }) func uploadUpiQrCode(blob : Storage.ExternalBlob) : async () {
    if (not (AccessControl.hasPermission(accessControlState, caller, #admin))) {
      Debug.trap("Unauthorized: Only admins can upload UPI QR code");
    };
    upiQrCode := ?blob;
  };

  public query func getUpiQrCode() : async ?Storage.ExternalBlob {
    upiQrCode;
  };
};

